// META: title=test WebNN API convTranspose2d operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-convtranspose2d
// Compute a 2-D transposed convolution given 4-D input and filter tensors.
//
// enum MLConvTranspose2dFilterOperandLayout {
//   "iohw",
//   "hwoi",
//   "ohwi"
// };
//
// dictionary MLConvTranspose2dOptions {
//   sequence<[EnforceRange] unsigned long> padding;
//   sequence<[EnforceRange] unsigned long> strides;
//   sequence<[EnforceRange] unsigned long> dilations;
//   sequence<[EnforceRange] unsigned long> outputPadding;
//   sequence<[EnforceRange] unsigned long> outputSizes;
//   [EnforceRange] unsigned long groups = 1;
//   MLInputOperandLayout inputLayout = "nchw";
//   MLConvTranspose2dFilterOperandLayout filterLayout = "iohw";
//   MLOperand bias;
// };
//
// MLOperand convTranspose2d(
//     MLOperand input, MLOperand filter,
//     optional MLConvTranspose2dOptions options = {});


const convTranspose2dTests = [
  {
    'name':
        'convTranspose2d float32 4D both input and filter non-constant tensors default options',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.5872158408164978, 0.6077792048454285, 0.017289165407419205,
            0.2614607512950897
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.3292713165283203, 0.5866857171058655, 0.29701370000839233,
            0.0033378428779542446
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'}, {'filter': 'convTranspose2dFilter'}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.1933533400297165, 0.5446354150772095, 0.3565753698348999,
            0.18010397255420685, 0.2787136137485504, 0.15542395412921906,
            0.0051351189613342285, 0.07771513611078262, 0.0008727149106562138
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D both input and filter constant tensors default options',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.5872158408164978, 0.6077792048454285, 0.017289165407419205,
            0.2614607512950897
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'},
          'constant': true
        },
        'convTranspose2dFilter': {
          'data': [
            0.3292713165283203, 0.5866857171058655, 0.29701370000839233,
            0.0033378428779542446
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'}, {'filter': 'convTranspose2dFilter'}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.1933533400297165, 0.5446354150772095, 0.3565753698348999,
            0.18010397255420685, 0.2787136137485504, 0.15542395412921906,
            0.0051351189613342285, 0.07771513611078262, 0.0008727149106562138
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors default options',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.5872158408164978, 0.6077792048454285, 0.017289165407419205,
            0.2614607512950897
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.3292713165283203, 0.5866857171058655, 0.29701370000839233,
            0.0033378428779542446
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'}, {'filter': 'convTranspose2dFilter'}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.1933533400297165, 0.5446354150772095, 0.3565753698348999,
            0.18010397255420685, 0.2787136137485504, 0.15542395412921906,
            0.0051351189613342285, 0.07771513611078262, 0.0008727149106562138
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.groups',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.8161798119544983, 0.5442776083946228, 0.7910669445991516,
            0.36564111709594727, 0.25429198145866394, 0.20815767347812653,
            0.7023073434829712, 0.5734469890594482
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.09232201427221298, 0.31896016001701355, 0.5445202589035034,
            0.6582807898521423, 0.9634373188018799, 0.012118860147893429,
            0.9230011701583862, 0.4781944155693054
          ],
          'descriptor': {shape: [2, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'}, {'options': {'groups': 2}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.07535136491060257, 0.3105776607990265, 0.1736028790473938,
            0.5174593329429626, 1.1197212934494019, 0.4749124348163605,
            0.4307519793510437, 0.7198431491851807, 0.24069452285766602,
            0.2449943870306015, 0.20362859964370728, 0.002522633643820882,
            0.9113409519195557, 0.8747221827507019, 0.10648936033248901,
            0.6482304930686951, 0.865131676197052, 0.2742191553115845
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.groups=2 options.strides=[2, 2]',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.8161798119544983, 0.5442776083946228, 0.7910669445991516,
            0.36564111709594727, 0.25429198145866394, 0.20815767347812653,
            0.7023073434829712, 0.5734469890594482
          ],
          'descriptor': {shape: [1, 2, 2, 2], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.09232201427221298, 0.31896016001701355, 0.5445202589035034,
            0.6582807898521423, 0.9634373188018799, 0.012118860147893429,
            0.9230011701583862, 0.4781944155693054
          ],
          'descriptor': {shape: [2, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'strides': [2, 2], 'groups': 2}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.07535136491060257, 0.26032882928848267,  0.050248805433511734,
            0.1736028790473938,  0.44442644715309143,  0.537275493144989,
            0.29637017846107483, 0.3582874834537506,   0.07303289324045181,
            0.2523188292980194,  0.03375672549009323,  0.11662495136260986,
            0.4307519793510437,  0.5207441449165344,   0.19909898936748505,
            0.24069452285766602, 0.2449943870306015,   0.0030817289371043444,
            0.20054687559604645, 0.002522633643820882, 0.23471179604530334,
            0.12160100787878036, 0.19212977588176727,  0.09953983873128891,
            0.6766291260719299,  0.008511164225637913, 0.5524802207946777,
            0.00694952392950654, 0.6482304930686951,   0.3358394503593445,
            0.5292922258377075,  0.2742191553115845
          ],
          'descriptor': {shape: [1, 2, 4, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.padding',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.5872158408164978, 0.6077792048454285, 0.017289165407419205,
            0.2614607512950897
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.3292713165283203, 0.5866857171058655, 0.29701370000839233,
            0.0033378428779542446
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'padding': [1, 1, 1, 1]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [0.2787136137485504],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'convTranspose2d options.padding is the same upper padding',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'}, {
            'options': {
              'outputSizes': [6, 6],
              'groups': 1,
              'strides': [2, 2],
              'dilations': [1, 1],
              'padding': [0, 1, 0, 1],
              'filterLayout': 'ohwi',
              'inputLayout': 'nhwc'
            }
          }
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5,
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5,
            1,   1,   1,   1,   2, 2, 1,   1,   2, 2, 1,   1,
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5,
            1,   1,   1,   1,   2, 2, 1,   1,   2, 2, 1,   1,
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5
          ],
          'descriptor': {shape: [1, 6, 6, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.strides',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'}, {'options': {'strides': [3, 2]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,  0.03513447195291519,   0.6485382318496704,
            0.4458966553211212,   1.015426516532898,     0.4092629551887512,
            0.4157154858112335,   0.0469902828335762,    0.0066622416488826275,
            0.6519761085510254,   0.08455146849155426,   1.2531912326812744,
            0.07760494202375412,  0.6478374600410461,    0.018416915088891983,
            0.04900681599974632,  0.27412328124046326,   0.6219525337219238,
            0.7271442413330078,   0.5708546042442322,    0.4705001711845398,
            0.3327140808105469,   0.24207575619220734,   0.5793426632881165,
            0.24261142313480377,  1.0615012645721436,    0.593023955821991,
            0.6023737192153931,   0.32376202940940857,   0.04590269923210144,
            0.7076690793037415,   0.0460042729973793,    1.177173137664795,
            0.11244992911815643,  0.9387195110321045,    0.12689214944839478,
            0.3376559019088745,   0.40547001361846924,   0.3384030759334564,
            0.5897663235664368,   0.8271709680557251,    0.6817569732666016,
            0.08246752619743347,  0.06000163406133652,   0.8564596176147461,
            0.5787978172302246,   1.1360399723052979,    0.39879822731018066,
            0.4050857424736023,   0.0802486464381218,    0.011377583257853985,
            0.8690866827964783,   0.1097523421049118,    1.4495694637298584,
            0.0756206065416336,   0.6312723755836487,    0.03145187348127365,
            0.08369242399930954,  0.37237587571144104,   0.8073278069496155,
            0.8744456768035889,   0.556257963180542,     0.45846959948539734,
            0.05494653806090355,  0.0034586030524224043, 0.7153855562210083,
            0.04389362782239914,  0.869132936000824,     0.04028744250535965,
            0.21026825904846191,  0.04203145205974579,   0.02203426882624626,
            0.5411697030067444,   0.2796400785446167,    0.5878635048866272,
            0.25666558742523193,  0.0901883915066719,    0.015911730006337166,
            0.023744819685816765, 0.21005792915821075,   0.30134889483451843,
            0.2883978486061096,   0.27659088373184204,   0.09458412230014801,
            0.3785804808139801,   0.02382970042526722,   0.5037901997566223,
            0.0238824300467968,   1.0520728826522827,    0.05837669596076012,
            0.3046796917915344,   0.2895958125591278,    0.15181563794612885,
            0.3435823321342468,   0.15215156972408295,   0.7628997564315796,
            0.37190964818000793,  0.13068340718746185,   0.1096314787864685,
            0.16360129415988922,  0.16581982374191284,   0.16396330296993256,
            0.3246387541294098,   0.400781512260437,     0.13705284893512726,
            0.09383610635995865,  0.00590650225058198,   0.9360047578811646,
            0.05697628855705261,  0.9210482239723206,    0.03925730288028717,
            0.20489174127578735,  0.07178010046482086,   0.03762948885560036,
            0.7056396007537842,   0.36298784613609314,   0.6046316623687744,
            0.2501027286052704,   0.08788229525089264,   0.027173593640327454,
            0.04055071249604225,  0.27599334716796875,   0.3911670744419098,
            0.3143731355667114,   0.26951852440834045,   0.09216563403606415
          ],
          'descriptor': {shape: [1, 2, 9, 7], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.dilations',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.3194596767425537, 0.9762163758277893, 0.4131408631801605,
            0.47982943058013916, 0.76741623878479, 0.9083173871040344,
            0.6205142140388489, 0.6580719947814941, 0.6553052067756653
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.6835425496101379, 0.9641214609146118, 0.8272836804389954,
            0.5771222710609436
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'dilations': [2, 2]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.21836428344249725, 0.6672854423522949,  0.590397298336029,
            0.9411911368370056,  0.39831796288490295, 0.3279838263988495,
            0.5245616436004639,  1.0834873914718628,  0.7398824691772461,
            0.8757283091545105,  0.6884316205978394,  1.2574280500411987,
            1.5723320245742798,  1.1978574991226196,  0.8702266216278076,
            0.39695504307746887, 0.6348709464073181,  1.0283564329147339,
            0.44289299845695496, 0.5242102146148682,  0.5133413076400757,
            0.5444121956825256,  0.9002358913421631,  0.37978801131248474,
            0.3781912326812744
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.outputPadding',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'strides': [3, 2], 'outputPadding': [1, 1]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,
            0.03513447195291519,
            0.6485382318496704,
            0.4458966553211212,
            1.015426516532898,
            0.4092629551887512,
            0.4157154858112335,
            0,
            0.0469902828335762,
            0.0066622416488826275,
            0.6519761085510254,
            0.08455146849155426,
            1.2531912326812744,
            0.07760494202375412,
            0.6478374600410461,
            0,
            0.018416915088891983,
            0.04900681599974632,
            0.27412328124046326,
            0.6219525337219238,
            0.7271442413330078,
            0.5708546042442322,
            0.4705001711845398,
            0,
            0.3327140808105469,
            0.24207575619220734,
            0.5793426632881165,
            0.24261142313480377,
            1.0615012645721436,
            0.593023955821991,
            0.6023737192153931,
            0,
            0.32376202940940857,
            0.04590269923210144,
            0.7076690793037415,
            0.0460042729973793,
            1.177173137664795,
            0.11244992911815643,
            0.9387195110321045,
            0,
            0.12689214944839478,
            0.3376559019088745,
            0.40547001361846924,
            0.3384030759334564,
            0.5897663235664368,
            0.8271709680557251,
            0.6817569732666016,
            0,
            0.08246752619743347,
            0.06000163406133652,
            0.8564596176147461,
            0.5787978172302246,
            1.1360399723052979,
            0.39879822731018066,
            0.4050857424736023,
            0,
            0.0802486464381218,
            0.011377583257853985,
            0.8690866827964783,
            0.1097523421049118,
            1.4495694637298584,
            0.0756206065416336,
            0.6312723755836487,
            0,
            0.03145187348127365,
            0.08369242399930954,
            0.37237587571144104,
            0.8073278069496155,
            0.8744456768035889,
            0.556257963180542,
            0.45846959948539734,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0.05494653806090355,
            0.0034586030524224043,
            0.7153855562210083,
            0.04389362782239914,
            0.869132936000824,
            0.04028744250535965,
            0.21026825904846191,
            0,
            0.04203145205974579,
            0.02203426882624626,
            0.5411697030067444,
            0.2796400785446167,
            0.5878635048866272,
            0.25666558742523193,
            0.0901883915066719,
            0,
            0.015911730006337166,
            0.023744819685816765,
            0.21005792915821075,
            0.30134889483451843,
            0.2883978486061096,
            0.27659088373184204,
            0.09458412230014801,
            0,
            0.3785804808139801,
            0.02382970042526722,
            0.5037901997566223,
            0.0238824300467968,
            1.0520728826522827,
            0.05837669596076012,
            0.3046796917915344,
            0,
            0.2895958125591278,
            0.15181563794612885,
            0.3435823321342468,
            0.15215156972408295,
            0.7628997564315796,
            0.37190964818000793,
            0.13068340718746185,
            0,
            0.1096314787864685,
            0.16360129415988922,
            0.16581982374191284,
            0.16396330296993256,
            0.3246387541294098,
            0.400781512260437,
            0.13705284893512726,
            0,
            0.09383610635995865,
            0.00590650225058198,
            0.9360047578811646,
            0.05697628855705261,
            0.9210482239723206,
            0.03925730288028717,
            0.20489174127578735,
            0,
            0.07178010046482086,
            0.03762948885560036,
            0.7056396007537842,
            0.36298784613609314,
            0.6046316623687744,
            0.2501027286052704,
            0.08788229525089264,
            0,
            0.027173593640327454,
            0.04055071249604225,
            0.27599334716796875,
            0.3911670744419098,
            0.3143731355667114,
            0.26951852440834045,
            0.09216563403606415,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0
          ],
          'descriptor': {shape: [1, 2, 10, 8], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.outputSizes',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'strides': [3, 2], 'outputSizes': [10, 8]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,
            0.03513447195291519,
            0.6485382318496704,
            0.4458966553211212,
            1.015426516532898,
            0.4092629551887512,
            0.4157154858112335,
            0,
            0.0469902828335762,
            0.0066622416488826275,
            0.6519761085510254,
            0.08455146849155426,
            1.2531912326812744,
            0.07760494202375412,
            0.6478374600410461,
            0,
            0.018416915088891983,
            0.04900681599974632,
            0.27412328124046326,
            0.6219525337219238,
            0.7271442413330078,
            0.5708546042442322,
            0.4705001711845398,
            0,
            0.3327140808105469,
            0.24207575619220734,
            0.5793426632881165,
            0.24261142313480377,
            1.0615012645721436,
            0.593023955821991,
            0.6023737192153931,
            0,
            0.32376202940940857,
            0.04590269923210144,
            0.7076690793037415,
            0.0460042729973793,
            1.177173137664795,
            0.11244992911815643,
            0.9387195110321045,
            0,
            0.12689214944839478,
            0.3376559019088745,
            0.40547001361846924,
            0.3384030759334564,
            0.5897663235664368,
            0.8271709680557251,
            0.6817569732666016,
            0,
            0.08246752619743347,
            0.06000163406133652,
            0.8564596176147461,
            0.5787978172302246,
            1.1360399723052979,
            0.39879822731018066,
            0.4050857424736023,
            0,
            0.0802486464381218,
            0.011377583257853985,
            0.8690866827964783,
            0.1097523421049118,
            1.4495694637298584,
            0.0756206065416336,
            0.6312723755836487,
            0,
            0.03145187348127365,
            0.08369242399930954,
            0.37237587571144104,
            0.8073278069496155,
            0.8744456768035889,
            0.556257963180542,
            0.45846959948539734,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0.05494653806090355,
            0.0034586030524224043,
            0.7153855562210083,
            0.04389362782239914,
            0.869132936000824,
            0.04028744250535965,
            0.21026825904846191,
            0,
            0.04203145205974579,
            0.02203426882624626,
            0.5411697030067444,
            0.2796400785446167,
            0.5878635048866272,
            0.25666558742523193,
            0.0901883915066719,
            0,
            0.015911730006337166,
            0.023744819685816765,
            0.21005792915821075,
            0.30134889483451843,
            0.2883978486061096,
            0.27659088373184204,
            0.09458412230014801,
            0,
            0.3785804808139801,
            0.02382970042526722,
            0.5037901997566223,
            0.0238824300467968,
            1.0520728826522827,
            0.05837669596076012,
            0.3046796917915344,
            0,
            0.2895958125591278,
            0.15181563794612885,
            0.3435823321342468,
            0.15215156972408295,
            0.7628997564315796,
            0.37190964818000793,
            0.13068340718746185,
            0,
            0.1096314787864685,
            0.16360129415988922,
            0.16581982374191284,
            0.16396330296993256,
            0.3246387541294098,
            0.400781512260437,
            0.13705284893512726,
            0,
            0.09383610635995865,
            0.00590650225058198,
            0.9360047578811646,
            0.05697628855705261,
            0.9210482239723206,
            0.03925730288028717,
            0.20489174127578735,
            0,
            0.07178010046482086,
            0.03762948885560036,
            0.7056396007537842,
            0.36298784613609314,
            0.6046316623687744,
            0.2501027286052704,
            0.08788229525089264,
            0,
            0.027173593640327454,
            0.04055071249604225,
            0.27599334716796875,
            0.3911670744419098,
            0.3143731355667114,
            0.26951852440834045,
            0.09216563403606415,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0
          ],
          'descriptor': {shape: [1, 2, 10, 8], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.inputLayout=nchw',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nchw'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,  0.6479843258857727,  1.0440847873687744,
            0.8621897101402283,   0.4157154858112335,  0.3797043561935425,
            1.1785486936569214,   1.9911006689071655,  1.6228916645050049,
            1.2502111196517944,   0.4246464669704437,  1.5086332559585571,
            3.287064790725708,    2.5666797161102295,  1.8143054246902466,
            0.20714078843593597,  1.2503143548965454,  1.6656538248062134,
            2.097904920578003,    1.313029408454895,   0.03145187348127365,
            0.38708874583244324,  1.0853508710861206,  1.2216601371765137,
            0.45846959948539734,  0.05494653806090355, 0.7007930278778076,
            0.7019880414009094,   0.26937708258628845, 0.21026825904846191,
            0.4206119179725647,   0.9587093591690063,  1.8526650667190552,
            0.5379507541656494,   0.39486807584762573, 0.3993436396121979,
            1.5788191556930542,   2.121230363845825,   1.141642689704895,
            0.4301592707633972,   0.18141157925128937, 1.0035220384597778,
            1.3417718410491943,   0.8345021605491638,  0.2249351441860199,
            0.027173593640327454, 0.3026771545410156,  0.5856420397758484,
            0.40328359603881836,  0.09216563403606415
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.inputLayout=nhwc',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196, 0.05494653806090355, 0.6479843258857727,
            0.7007930278778076,  1.0440847873687744,  0.7019880414009094,
            0.8621897101402283,  0.26937708258628845, 0.4157154858112335,
            0.21026825904846191, 0.3797043561935425,  0.4206119179725647,
            1.1785486936569214,  0.9587093591690063,  1.9911006689071655,
            1.8526650667190552,  1.6228916645050049,  0.5379507541656494,
            1.2502111196517944,  0.39486807584762573, 0.4246464669704437,
            0.3993436396121979,  1.5086332559585571,  1.5788191556930542,
            3.287064790725708,   2.121230363845825,   2.5666797161102295,
            1.141642689704895,   1.8143054246902466,  0.4301592707633972,
            0.20714078843593597, 0.18141157925128937, 1.2503143548965454,
            1.0035220384597778,  1.6656538248062134,  1.3417718410491943,
            2.097904920578003,   0.8345021605491638,  1.313029408454895,
            0.2249351441860199,  0.03145187348127365, 0.027173593640327454,
            0.38708874583244324, 0.3026771545410156,  1.0853508710861206,
            0.5856420397758484,  1.2216601371765137,  0.40328359603881836,
            0.45846959948539734, 0.09216563403606415
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.filterLayout=iohw',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'filterLayout': 'iohw'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,  0.6479843258857727,  1.0440847873687744,
            0.8621897101402283,   0.4157154858112335,  0.3797043561935425,
            1.1785486936569214,   1.9911006689071655,  1.6228916645050049,
            1.2502111196517944,   0.4246464669704437,  1.5086332559585571,
            3.287064790725708,    2.5666797161102295,  1.8143054246902466,
            0.20714078843593597,  1.2503143548965454,  1.6656538248062134,
            2.097904920578003,    1.313029408454895,   0.03145187348127365,
            0.38708874583244324,  1.0853508710861206,  1.2216601371765137,
            0.45846959948539734,  0.05494653806090355, 0.7007930278778076,
            0.7019880414009094,   0.26937708258628845, 0.21026825904846191,
            0.4206119179725647,   0.9587093591690063,  1.8526650667190552,
            0.5379507541656494,   0.39486807584762573, 0.3993436396121979,
            1.5788191556930542,   2.121230363845825,   1.141642689704895,
            0.4301592707633972,   0.18141157925128937, 1.0035220384597778,
            1.3417718410491943,   0.8345021605491638,  0.2249351441860199,
            0.027173593640327454, 0.3026771545410156,  0.5856420397758484,
            0.40328359603881836,  0.09216563403606415
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.filterLayout=hwoi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.9801966547966003, 0.6267672777175903,
            0.06169835478067398, 0.6366490125656128, 0.3220160901546478,
            0.8382642269134521, 0.7498031854629517, 0.11884837597608566,
            0.3930714726448059, 0.9921330213546753, 0.13811933994293213,
            0.3285411298274994, 0.28385090827941895, 0.8742373585700989,
            0.4235861301422119, 0.7205492258071899, 0.1448512077331543
          ],
          'descriptor': {shape: [3, 3, 2, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'filterLayout': 'hwoi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,  0.6479843258857727,  1.0440847873687744,
            0.8621897101402283,   0.4157154858112335,  0.3797043561935425,
            1.1785486936569214,   1.9911006689071655,  1.6228916645050049,
            1.2502111196517944,   0.4246464669704437,  1.5086332559585571,
            3.287064790725708,    2.5666797161102295,  1.8143054246902466,
            0.20714078843593597,  1.2503143548965454,  1.6656538248062134,
            2.097904920578003,    1.313029408454895,   0.03145187348127365,
            0.38708874583244324,  1.0853508710861206,  1.2216601371765137,
            0.45846959948539734,  0.05494653806090355, 0.7007930278778076,
            0.7019880414009094,   0.26937708258628845, 0.21026825904846191,
            0.4206119179725647,   0.9587093591690063,  1.8526650667190552,
            0.5379507541656494,   0.39486807584762573, 0.3993436396121979,
            1.5788191556930542,   2.121230363845825,   1.141642689704895,
            0.4301592707633972,   0.18141157925128937, 1.0035220384597778,
            1.3417718410491943,   0.8345021605491638,  0.2249351441860199,
            0.027173593640327454, 0.3026771545410156,  0.5856420397758484,
            0.40328359603881836,  0.09216563403606415
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.filterLayout=ohwi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'filterLayout': 'ohwi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196,  0.6479843258857727,  1.0440847873687744,
            0.8621897101402283,   0.4157154858112335,  0.3797043561935425,
            1.1785486936569214,   1.9911006689071655,  1.6228916645050049,
            1.2502111196517944,   0.4246464669704437,  1.5086332559585571,
            3.287064790725708,    2.5666797161102295,  1.8143054246902466,
            0.20714078843593597,  1.2503143548965454,  1.6656538248062134,
            2.097904920578003,    1.313029408454895,   0.03145187348127365,
            0.38708874583244324,  1.0853508710861206,  1.2216601371765137,
            0.45846959948539734,  0.05494653806090355, 0.7007930278778076,
            0.7019880414009094,   0.26937708258628845, 0.21026825904846191,
            0.4206119179725647,   0.9587093591690063,  1.8526650667190552,
            0.5379507541656494,   0.39486807584762573, 0.3993436396121979,
            1.5788191556930542,   2.121230363845825,   1.141642689704895,
            0.4301592707633972,   0.18141157925128937, 1.0035220384597778,
            1.3417718410491943,   0.8345021605491638,  0.2249351441860199,
            0.027173593640327454, 0.3026771545410156,  0.5856420397758484,
            0.40328359603881836,  0.09216563403606415
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.inputLayout=nhwc options.filterLayout=iohw',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'iohw'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196, 0.05494653806090355, 0.6479843258857727,
            0.7007930278778076,  1.0440847873687744,  0.7019880414009094,
            0.8621897101402283,  0.26937708258628845, 0.4157154858112335,
            0.21026825904846191, 0.3797043561935425,  0.4206119179725647,
            1.1785486936569214,  0.9587093591690063,  1.9911006689071655,
            1.8526650667190552,  1.6228916645050049,  0.5379507541656494,
            1.2502111196517944,  0.39486807584762573, 0.4246464669704437,
            0.3993436396121979,  1.5086332559585571,  1.5788191556930542,
            3.287064790725708,   2.121230363845825,   2.5666797161102295,
            1.141642689704895,   1.8143054246902466,  0.4301592707633972,
            0.20714078843593597, 0.18141157925128937, 1.2503143548965454,
            1.0035220384597778,  1.6656538248062134,  1.3417718410491943,
            2.097904920578003,   0.8345021605491638,  1.313029408454895,
            0.2249351441860199,  0.03145187348127365, 0.027173593640327454,
            0.38708874583244324, 0.3026771545410156,  1.0853508710861206,
            0.5856420397758484,  1.2216601371765137,  0.40328359603881836,
            0.45846959948539734, 0.09216563403606415
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.inputLayout=nhwc options.filterLayout=hwoi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.9801966547966003, 0.6267672777175903,
            0.06169835478067398, 0.6366490125656128, 0.3220160901546478,
            0.8382642269134521, 0.7498031854629517, 0.11884837597608566,
            0.3930714726448059, 0.9921330213546753, 0.13811933994293213,
            0.3285411298274994, 0.28385090827941895, 0.8742373585700989,
            0.4235861301422119, 0.7205492258071899, 0.1448512077331543
          ],
          'descriptor': {shape: [3, 3, 2, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'hwoi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196, 0.05494653806090355, 0.6479843258857727,
            0.7007930278778076,  1.0440847873687744,  0.7019880414009094,
            0.8621897101402283,  0.26937708258628845, 0.4157154858112335,
            0.21026825904846191, 0.3797043561935425,  0.4206119179725647,
            1.1785486936569214,  0.9587093591690063,  1.9911006689071655,
            1.8526650667190552,  1.6228916645050049,  0.5379507541656494,
            1.2502111196517944,  0.39486807584762573, 0.4246464669704437,
            0.3993436396121979,  1.5086332559585571,  1.5788191556930542,
            3.287064790725708,   2.121230363845825,   2.5666797161102295,
            1.141642689704895,   1.8143054246902466,  0.4301592707633972,
            0.20714078843593597, 0.18141157925128937, 1.2503143548965454,
            1.0035220384597778,  1.6656538248062134,  1.3417718410491943,
            2.097904920578003,   0.8345021605491638,  1.313029408454895,
            0.2249351441860199,  0.03145187348127365, 0.027173593640327454,
            0.38708874583244324, 0.3026771545410156,  1.0853508710861206,
            0.5856420397758484,  1.2216601371765137,  0.40328359603881836,
            0.45846959948539734, 0.09216563403606415
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors options.inputLayout=nhwc options.filterLayout=ohwi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.05605664849281311, 0.7114229798316956, 0.6529743671417236,
            0.38622909784317017, 0.3870837390422821, 0.9461629390716553,
            0.09573192149400711, 0.9234652519226074, 0.636277973651886
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.8614422678947449, 0.6267672777175903, 0.6366490125656128,
            0.8382642269134521, 0.11884837597608566, 0.9921330213546753,
            0.3285411298274994, 0.8742373585700989, 0.7205492258071899,
            0.9801966547966003, 0.06169835478067398, 0.3220160901546478,
            0.7498031854629517, 0.3930714726448059, 0.13811933994293213,
            0.28385090827941895, 0.4235861301422119, 0.1448512077331543
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'ohwi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04828956723213196, 0.05494653806090355, 0.6479843258857727,
            0.7007930278778076,  1.0440847873687744,  0.7019880414009094,
            0.8621897101402283,  0.26937708258628845, 0.4157154858112335,
            0.21026825904846191, 0.3797043561935425,  0.4206119179725647,
            1.1785486936569214,  0.9587093591690063,  1.9911006689071655,
            1.8526650667190552,  1.6228916645050049,  0.5379507541656494,
            1.2502111196517944,  0.39486807584762573, 0.4246464669704437,
            0.3993436396121979,  1.5086332559585571,  1.5788191556930542,
            3.287064790725708,   2.121230363845825,   2.5666797161102295,
            1.141642689704895,   1.8143054246902466,  0.4301592707633972,
            0.20714078843593597, 0.18141157925128937, 1.2503143548965454,
            1.0035220384597778,  1.6656538248062134,  1.3417718410491943,
            2.097904920578003,   0.8345021605491638,  1.313029408454895,
            0.2249351441860199,  0.03145187348127365, 0.027173593640327454,
            0.38708874583244324, 0.3026771545410156,  1.0853508710861206,
            0.5856420397758484,  1.2216601371765137,  0.40328359603881836,
            0.45846959948539734, 0.09216563403606415
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'convTranspose2d float32 4D input and filter tensors options.bias',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.1109575480222702, 0.8681362271308899, 0.7342095971107483,
            0.43077003955841064, 0.5981627106666565, 0.12321650236845016,
            0.1610974818468094, 0.0884026437997818, 0.29100972414016724
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.6161394715309143, 0.26224616169929504, 0.7951397895812988,
            0.8730561137199402, 0.8309102058410645, 0.854960560798645,
            0.5552039742469788, 0.840092122554779, 0.85308438539505
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        },
        'convTranspose2dBias': {
          'data': [0.451673686504364],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'bias': 'convTranspose2dBias'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.5200390219688416, 1.01566481590271,   1.2199413776397705,
            1.3345069885253906, 1.0354729890823364, 0.8139602541923523,
            1.7833205461502075, 2.484194278717041,  2.311894178390503,
            1.1773682832717896, 0.9886226654052734, 2.0037572383880615,
            2.9867470264434814, 2.5694668292999268, 1.41475510597229,
            0.8314860463142395, 1.3567005395889282, 1.8553334474563599,
            1.3828538656234741, 0.8055896162986755, 0.5411156415939331,
            0.6360918879508972, 0.8249395489692688, 0.7715635895729065,
            0.6999295353889465
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float32 4D input and filter tensors, both negative input tensor and options.bias',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            -0.10889056324958801, -0.29801905155181885, -0.3907785713672638,
            -0.5624061226844788, -0.7322093844413757, -0.8421320915222168,
            -0.30598655343055725, -0.976659893989563, -0.014158561825752258
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.6161394715309143, 0.26224616169929504, 0.7951397895812988,
            0.8730561137199402, 0.8309102058410645, 0.854960560798645,
            0.5552039742469788, 0.840092122554779, 0.85308438539505
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        },
        'convTranspose2dBias': {
          'data': [-0.8457866311073303],
          'descriptor': {shape: [1], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'bias': 'convTranspose2dBias'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            -0.9128783941268921, -1.0579640865325928, -1.2512983083724976,
            -1.1852335929870605, -1.1565102338790894, -1.2873748540878296,
            -1.7950842380523682, -2.6857638359069824, -2.2283377647399902,
            -1.8494995832443237, -1.5857856273651123, -2.8912975788116455,
            -3.738619565963745,  -3.5343525409698486, -1.910401463508606,
            -1.425180196762085,  -2.8317112922668457, -3.49372935295105,
            -3.0246617794036865, -1.5763013362884521, -1.0156716108322144,
            -1.645089030265808,  -1.935164213180542,  -1.6908544301986694,
            -0.8578650951385498
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'convTranspose2d same output size different padding (padding=1, outputPadding=0))',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            1, 2, 3,
            4, 5, 6,
            7, 8, 9
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            1, 1, 1,
            1, 1, 1,
            1, 1, 1
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {
            'strides': [3, 3],
            'padding': [1, 1, 1, 1],
            'outputPadding': [0, 0]
          }}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            1, 1, 2, 2, 2, 3, 3,
            1, 1, 2, 2, 2, 3, 3,
            4, 4, 5, 5, 5, 6, 6,
            4, 4, 5, 5, 5, 6, 6,
            4, 4, 5, 5, 5, 6, 6,
            7, 7, 8, 8, 8, 9, 9,
            7, 7, 8, 8, 8, 9, 9
          ],
          'descriptor': {shape: [1, 1, 7, 7], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'convTranspose2d same output size different padding (padding=2, outputPadding=2))',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            1, 2, 3,
            4, 5, 6,
            7, 8, 9
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'}
        },
        'convTranspose2dFilter': {
          'data': [
            1, 1, 1,
            1, 1, 1,
            1, 1, 1
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {
            'strides': [3, 3],
            'padding': [2, 2, 2, 2],
            'outputPadding': [2, 2]
          }}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            1, 2, 2, 2, 3, 3, 3,
            4, 5, 5, 5, 6, 6, 6,
            4, 5, 5, 5, 6, 6, 6,
            4, 5, 5, 5, 6, 6, 6,
            7, 8, 8, 8, 9, 9, 9,
            7, 8, 8, 8, 9, 9, 9,
            7, 8, 8, 8, 9, 9, 9
          ],
          'descriptor': {shape: [1, 1, 7, 7], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name':
        'convTranspose2d float16 4D both input and filter non-constant tensors default options',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.58740234375, 0.60791015625, 0.0172882080078125, 0.261474609375
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.329345703125, 0.5869140625, 0.297119140625, 0.003337860107421875
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'}, {'filter': 'convTranspose2dFilter'}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.1934814453125, 0.544921875, 0.356689453125, 0.18017578125,
            0.27880859375, 0.155517578125, 0.005138397216796875,
            0.0777587890625, 0.0008726119995117188
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D both input and filter constant tensors default options',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.58740234375, 0.60791015625, 0.0172882080078125, 0.261474609375
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'},
          'constant': true
        },
        'convTranspose2dFilter': {
          'data': [
            0.329345703125, 0.5869140625, 0.297119140625, 0.003337860107421875
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'}, {'filter': 'convTranspose2dFilter'}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.1934814453125, 0.544921875, 0.356689453125, 0.18017578125,
            0.27880859375, 0.155517578125, 0.005138397216796875,
            0.0777587890625, 0.0008726119995117188
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors default options',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.58740234375, 0.60791015625, 0.0172882080078125, 0.261474609375
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.329345703125, 0.5869140625, 0.297119140625, 0.003337860107421875
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'}, {'filter': 'convTranspose2dFilter'}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.1934814453125, 0.544921875, 0.356689453125, 0.18017578125,
            0.27880859375, 0.155517578125, 0.005138397216796875,
            0.0777587890625, 0.0008726119995117188
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.padding',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.58740234375, 0.60791015625, 0.0172882080078125, 0.261474609375
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.329345703125, 0.5869140625, 0.297119140625, 0.003337860107421875
          ],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'padding': [1, 1, 1, 1]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [0.27880859375],
          'descriptor': {shape: [1, 1, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 input tensors options.padding is the same upper padding',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5, 0.5],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'}, {
            'options': {
              'outputSizes': [6, 6],
              'groups': 1,
              'strides': [2, 2],
              'dilations': [1, 1],
              'padding': [0, 1, 0, 1],
              'filterLayout': 'ohwi',
              'inputLayout': 'nhwc'
            }
          }
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5,
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5,
            1,   1,   1,   1,   2, 2, 1,   1,   2, 2, 1,   1,
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5,
            1,   1,   1,   1,   2, 2, 1,   1,   2, 2, 1,   1,
            0.5, 0.5, 0.5, 0.5, 1, 1, 0.5, 0.5, 1, 1, 0.5, 0.5
          ],
          'descriptor': {shape: [1, 6, 6, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.strides',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'}, {'options': {'strides': [3, 2]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.03515625,
            0.6484375,
            0.446044921875,
            1.015625,
            0.4091796875,
            0.415771484375,
            0.0469970703125,
            0.00666046142578125,
            0.65185546875,
            0.08453369140625,
            1.2529296875,
            0.07757568359375,
            0.64794921875,
            0.0184173583984375,
            0.04901123046875,
            0.274169921875,
            0.62158203125,
            0.72705078125,
            0.57080078125,
            0.470458984375,
            0.332763671875,
            0.2421875,
            0.5791015625,
            0.2425537109375,
            1.0615234375,
            0.59326171875,
            0.6025390625,
            0.32373046875,
            0.0458984375,
            0.70751953125,
            0.045989990234375,
            1.177734375,
            0.1124267578125,
            0.93896484375,
            0.126953125,
            0.337646484375,
            0.405517578125,
            0.338134765625,
            0.58984375,
            0.8271484375,
            0.68212890625,
            0.08245849609375,
            0.05999755859375,
            0.8564453125,
            0.5791015625,
            1.1357421875,
            0.39892578125,
            0.405029296875,
            0.08026123046875,
            0.01137542724609375,
            0.869140625,
            0.1097412109375,
            1.44921875,
            0.07562255859375,
            0.63134765625,
            0.031463623046875,
            0.0836181640625,
            0.372314453125,
            0.80712890625,
            0.87451171875,
            0.55615234375,
            0.45849609375,
            0.054931640625,
            0.003459930419921875,
            0.71533203125,
            0.043914794921875,
            0.86865234375,
            0.040283203125,
            0.210205078125,
            0.04205322265625,
            0.02203369140625,
            0.54150390625,
            0.279541015625,
            0.587890625,
            0.256591796875,
            0.09014892578125,
            0.0159149169921875,
            0.02374267578125,
            0.2100830078125,
            0.30126953125,
            0.288330078125,
            0.276611328125,
            0.0946044921875,
            0.37841796875,
            0.023834228515625,
            0.50341796875,
            0.0238800048828125,
            1.0517578125,
            0.058380126953125,
            0.3046875,
            0.28955078125,
            0.15185546875,
            0.343505859375,
            0.152099609375,
            0.76318359375,
            0.3720703125,
            0.130615234375,
            0.10968017578125,
            0.16357421875,
            0.1658935546875,
            0.1639404296875,
            0.32470703125,
            0.40087890625,
            0.1370849609375,
            0.09381103515625,
            0.0059051513671875,
            0.935546875,
            0.056976318359375,
            0.9208984375,
            0.03924560546875,
            0.204833984375,
            0.07177734375,
            0.037628173828125,
            0.70556640625,
            0.363037109375,
            0.6044921875,
            0.25,
            0.08782958984375,
            0.0271759033203125,
            0.04052734375,
            0.276123046875,
            0.39111328125,
            0.314453125,
            0.26953125,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 2, 9, 7], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.dilations',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.319580078125, 0.97607421875, 0.4130859375, 0.479736328125,
            0.767578125, 0.908203125, 0.62060546875, 0.658203125, 0.6552734375
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [0.68359375, 0.96435546875, 0.8271484375, 0.5771484375],
          'descriptor': {shape: [1, 1, 2, 2], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'dilations': [2, 2]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.218505859375, 0.66748046875,  0.59033203125,  0.94140625,
            0.3984375,      0.327880859375, 0.52490234375,  1.0830078125,
            0.740234375,    0.8759765625,   0.6884765625,   1.2568359375,
            1.572265625,    1.1982421875,   0.8701171875,   0.396728515625,
            0.634765625,    1.0283203125,   0.443115234375, 0.52392578125,
            0.51318359375,  0.54443359375,  0.900390625,    0.3798828125,
            0.378173828125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.outputPadding',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'strides': [3, 2], 'outputPadding': [1, 1]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.03515625,
            0.6484375,
            0.446044921875,
            1.015625,
            0.4091796875,
            0.415771484375,
            0,
            0.0469970703125,
            0.00666046142578125,
            0.65185546875,
            0.08453369140625,
            1.2529296875,
            0.07757568359375,
            0.64794921875,
            0,
            0.0184173583984375,
            0.04901123046875,
            0.274169921875,
            0.62158203125,
            0.72705078125,
            0.57080078125,
            0.470458984375,
            0,
            0.332763671875,
            0.2421875,
            0.5791015625,
            0.2425537109375,
            1.0615234375,
            0.59326171875,
            0.6025390625,
            0,
            0.32373046875,
            0.0458984375,
            0.70751953125,
            0.045989990234375,
            1.177734375,
            0.1124267578125,
            0.93896484375,
            0,
            0.126953125,
            0.337646484375,
            0.405517578125,
            0.338134765625,
            0.58984375,
            0.8271484375,
            0.68212890625,
            0,
            0.08245849609375,
            0.05999755859375,
            0.8564453125,
            0.5791015625,
            1.1357421875,
            0.39892578125,
            0.405029296875,
            0,
            0.08026123046875,
            0.01137542724609375,
            0.869140625,
            0.1097412109375,
            1.44921875,
            0.07562255859375,
            0.63134765625,
            0,
            0.031463623046875,
            0.0836181640625,
            0.372314453125,
            0.80712890625,
            0.87451171875,
            0.55615234375,
            0.45849609375,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0.054931640625,
            0.003459930419921875,
            0.71533203125,
            0.043914794921875,
            0.86865234375,
            0.040283203125,
            0.210205078125,
            0,
            0.04205322265625,
            0.02203369140625,
            0.54150390625,
            0.279541015625,
            0.587890625,
            0.256591796875,
            0.09014892578125,
            0,
            0.0159149169921875,
            0.02374267578125,
            0.2100830078125,
            0.30126953125,
            0.288330078125,
            0.276611328125,
            0.0946044921875,
            0,
            0.37841796875,
            0.023834228515625,
            0.50341796875,
            0.0238800048828125,
            1.0517578125,
            0.058380126953125,
            0.3046875,
            0,
            0.28955078125,
            0.15185546875,
            0.343505859375,
            0.152099609375,
            0.76318359375,
            0.3720703125,
            0.130615234375,
            0,
            0.10968017578125,
            0.16357421875,
            0.1658935546875,
            0.1639404296875,
            0.32470703125,
            0.40087890625,
            0.1370849609375,
            0,
            0.09381103515625,
            0.0059051513671875,
            0.935546875,
            0.056976318359375,
            0.9208984375,
            0.03924560546875,
            0.204833984375,
            0,
            0.07177734375,
            0.037628173828125,
            0.70556640625,
            0.363037109375,
            0.6044921875,
            0.25,
            0.08782958984375,
            0,
            0.0271759033203125,
            0.04052734375,
            0.276123046875,
            0.39111328125,
            0.314453125,
            0.26953125,
            0.0921630859375,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0
          ],
          'descriptor': {shape: [1, 2, 10, 8], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.outputSizes',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'strides': [3, 2], 'outputSizes': [10, 8]}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.03515625,
            0.6484375,
            0.446044921875,
            1.015625,
            0.4091796875,
            0.415771484375,
            0,
            0.0469970703125,
            0.00666046142578125,
            0.65185546875,
            0.08453369140625,
            1.2529296875,
            0.07757568359375,
            0.64794921875,
            0,
            0.0184173583984375,
            0.04901123046875,
            0.274169921875,
            0.62158203125,
            0.72705078125,
            0.57080078125,
            0.470458984375,
            0,
            0.332763671875,
            0.2421875,
            0.5791015625,
            0.2425537109375,
            1.0615234375,
            0.59326171875,
            0.6025390625,
            0,
            0.32373046875,
            0.0458984375,
            0.70751953125,
            0.045989990234375,
            1.177734375,
            0.1124267578125,
            0.93896484375,
            0,
            0.126953125,
            0.337646484375,
            0.405517578125,
            0.338134765625,
            0.58984375,
            0.8271484375,
            0.68212890625,
            0,
            0.08245849609375,
            0.05999755859375,
            0.8564453125,
            0.5791015625,
            1.1357421875,
            0.39892578125,
            0.405029296875,
            0,
            0.08026123046875,
            0.01137542724609375,
            0.869140625,
            0.1097412109375,
            1.44921875,
            0.07562255859375,
            0.63134765625,
            0,
            0.031463623046875,
            0.0836181640625,
            0.372314453125,
            0.80712890625,
            0.87451171875,
            0.55615234375,
            0.45849609375,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0.054931640625,
            0.003459930419921875,
            0.71533203125,
            0.043914794921875,
            0.86865234375,
            0.040283203125,
            0.210205078125,
            0,
            0.04205322265625,
            0.02203369140625,
            0.54150390625,
            0.279541015625,
            0.587890625,
            0.256591796875,
            0.09014892578125,
            0,
            0.0159149169921875,
            0.02374267578125,
            0.2100830078125,
            0.30126953125,
            0.288330078125,
            0.276611328125,
            0.0946044921875,
            0,
            0.37841796875,
            0.023834228515625,
            0.50341796875,
            0.0238800048828125,
            1.0517578125,
            0.058380126953125,
            0.3046875,
            0,
            0.28955078125,
            0.15185546875,
            0.343505859375,
            0.152099609375,
            0.76318359375,
            0.3720703125,
            0.130615234375,
            0,
            0.10968017578125,
            0.16357421875,
            0.1658935546875,
            0.1639404296875,
            0.32470703125,
            0.40087890625,
            0.1370849609375,
            0,
            0.09381103515625,
            0.0059051513671875,
            0.935546875,
            0.056976318359375,
            0.9208984375,
            0.03924560546875,
            0.204833984375,
            0,
            0.07177734375,
            0.037628173828125,
            0.70556640625,
            0.363037109375,
            0.6044921875,
            0.25,
            0.08782958984375,
            0,
            0.0271759033203125,
            0.04052734375,
            0.276123046875,
            0.39111328125,
            0.314453125,
            0.26953125,
            0.0921630859375,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0,
            0
          ],
          'descriptor': {shape: [1, 2, 10, 8], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.inputLayout=nchw',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nchw'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.64794921875,
            1.0439453125,
            0.8623046875,
            0.415771484375,
            0.379638671875,
            1.1787109375,
            1.9912109375,
            1.623046875,
            1.25,
            0.424560546875,
            1.5087890625,
            3.287109375,
            2.56640625,
            1.814453125,
            0.2071533203125,
            1.25,
            1.666015625,
            2.09765625,
            1.3134765625,
            0.031463623046875,
            0.386962890625,
            1.0849609375,
            1.2216796875,
            0.45849609375,
            0.054931640625,
            0.70068359375,
            0.70166015625,
            0.269287109375,
            0.210205078125,
            0.420654296875,
            0.95849609375,
            1.8525390625,
            0.53759765625,
            0.394775390625,
            0.3994140625,
            1.578125,
            2.12109375,
            1.1416015625,
            0.43017578125,
            0.181396484375,
            1.00390625,
            1.341796875,
            0.83447265625,
            0.2249755859375,
            0.0271759033203125,
            0.302734375,
            0.58544921875,
            0.4033203125,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.inputLayout=nhwc',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.054931640625,
            0.64794921875,
            0.70068359375,
            1.0439453125,
            0.70166015625,
            0.8623046875,
            0.269287109375,
            0.415771484375,
            0.210205078125,
            0.379638671875,
            0.420654296875,
            1.1787109375,
            0.95849609375,
            1.9912109375,
            1.8525390625,
            1.623046875,
            0.53759765625,
            1.25,
            0.394775390625,
            0.424560546875,
            0.3994140625,
            1.5087890625,
            1.578125,
            3.287109375,
            2.12109375,
            2.56640625,
            1.1416015625,
            1.814453125,
            0.43017578125,
            0.2071533203125,
            0.181396484375,
            1.25,
            1.00390625,
            1.666015625,
            1.341796875,
            2.09765625,
            0.83447265625,
            1.3134765625,
            0.2249755859375,
            0.031463623046875,
            0.0271759033203125,
            0.386962890625,
            0.302734375,
            1.0849609375,
            0.58544921875,
            1.2216796875,
            0.4033203125,
            0.45849609375,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.filterLayout=iohw',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'filterLayout': 'iohw'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.64794921875,
            1.0439453125,
            0.8623046875,
            0.415771484375,
            0.379638671875,
            1.1787109375,
            1.9912109375,
            1.623046875,
            1.25,
            0.424560546875,
            1.5087890625,
            3.287109375,
            2.56640625,
            1.814453125,
            0.2071533203125,
            1.25,
            1.666015625,
            2.09765625,
            1.3134765625,
            0.031463623046875,
            0.386962890625,
            1.0849609375,
            1.2216796875,
            0.45849609375,
            0.054931640625,
            0.70068359375,
            0.70166015625,
            0.269287109375,
            0.210205078125,
            0.420654296875,
            0.95849609375,
            1.8525390625,
            0.53759765625,
            0.394775390625,
            0.3994140625,
            1.578125,
            2.12109375,
            1.1416015625,
            0.43017578125,
            0.181396484375,
            1.00390625,
            1.341796875,
            0.83447265625,
            0.2249755859375,
            0.0271759033203125,
            0.302734375,
            0.58544921875,
            0.4033203125,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.filterLayout=hwoi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.97998046875, 0.626953125, 0.06170654296875,
            0.63671875, 0.322021484375, 0.83837890625, 0.75, 0.11883544921875,
            0.39306640625, 0.9921875, 0.1380615234375, 0.32861328125,
            0.283935546875, 0.8740234375, 0.423583984375, 0.720703125,
            0.1448974609375
          ],
          'descriptor': {shape: [3, 3, 2, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'filterLayout': 'hwoi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.64794921875,
            1.0439453125,
            0.8623046875,
            0.415771484375,
            0.379638671875,
            1.1787109375,
            1.9912109375,
            1.623046875,
            1.25,
            0.424560546875,
            1.5087890625,
            3.287109375,
            2.56640625,
            1.814453125,
            0.2071533203125,
            1.25,
            1.666015625,
            2.09765625,
            1.3134765625,
            0.031463623046875,
            0.386962890625,
            1.0849609375,
            1.2216796875,
            0.45849609375,
            0.054931640625,
            0.70068359375,
            0.70166015625,
            0.269287109375,
            0.210205078125,
            0.420654296875,
            0.95849609375,
            1.8525390625,
            0.53759765625,
            0.394775390625,
            0.3994140625,
            1.578125,
            2.12109375,
            1.1416015625,
            0.43017578125,
            0.181396484375,
            1.00390625,
            1.341796875,
            0.83447265625,
            0.2249755859375,
            0.0271759033203125,
            0.302734375,
            0.58544921875,
            0.4033203125,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.filterLayout=ohwi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'filterLayout': 'ohwi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.64794921875,
            1.0439453125,
            0.8623046875,
            0.415771484375,
            0.379638671875,
            1.1787109375,
            1.9912109375,
            1.623046875,
            1.25,
            0.424560546875,
            1.5087890625,
            3.287109375,
            2.56640625,
            1.814453125,
            0.2071533203125,
            1.25,
            1.666015625,
            2.09765625,
            1.3134765625,
            0.031463623046875,
            0.386962890625,
            1.0849609375,
            1.2216796875,
            0.45849609375,
            0.054931640625,
            0.70068359375,
            0.70166015625,
            0.269287109375,
            0.210205078125,
            0.420654296875,
            0.95849609375,
            1.8525390625,
            0.53759765625,
            0.394775390625,
            0.3994140625,
            1.578125,
            2.12109375,
            1.1416015625,
            0.43017578125,
            0.181396484375,
            1.00390625,
            1.341796875,
            0.83447265625,
            0.2249755859375,
            0.0271759033203125,
            0.302734375,
            0.58544921875,
            0.4033203125,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 2, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.inputLayout=nhwc options.filterLayout=iohw',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [1, 2, 3, 3], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'iohw'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.054931640625,
            0.64794921875,
            0.70068359375,
            1.0439453125,
            0.70166015625,
            0.8623046875,
            0.269287109375,
            0.415771484375,
            0.210205078125,
            0.379638671875,
            0.420654296875,
            1.1787109375,
            0.95849609375,
            1.9912109375,
            1.8525390625,
            1.623046875,
            0.53759765625,
            1.25,
            0.394775390625,
            0.424560546875,
            0.3994140625,
            1.5087890625,
            1.578125,
            3.287109375,
            2.12109375,
            2.56640625,
            1.1416015625,
            1.814453125,
            0.43017578125,
            0.2071533203125,
            0.181396484375,
            1.25,
            1.00390625,
            1.666015625,
            1.341796875,
            2.09765625,
            0.83447265625,
            1.3134765625,
            0.2249755859375,
            0.031463623046875,
            0.0271759033203125,
            0.386962890625,
            0.302734375,
            1.0849609375,
            0.58544921875,
            1.2216796875,
            0.4033203125,
            0.45849609375,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.inputLayout=nhwc options.filterLayout=hwoi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.97998046875, 0.626953125, 0.06170654296875,
            0.63671875, 0.322021484375, 0.83837890625, 0.75, 0.11883544921875,
            0.39306640625, 0.9921875, 0.1380615234375, 0.32861328125,
            0.283935546875, 0.8740234375, 0.423583984375, 0.720703125,
            0.1448974609375
          ],
          'descriptor': {shape: [3, 3, 2, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'hwoi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.054931640625,
            0.64794921875,
            0.70068359375,
            1.0439453125,
            0.70166015625,
            0.8623046875,
            0.269287109375,
            0.415771484375,
            0.210205078125,
            0.379638671875,
            0.420654296875,
            1.1787109375,
            0.95849609375,
            1.9912109375,
            1.8525390625,
            1.623046875,
            0.53759765625,
            1.25,
            0.394775390625,
            0.424560546875,
            0.3994140625,
            1.5087890625,
            1.578125,
            3.287109375,
            2.12109375,
            2.56640625,
            1.1416015625,
            1.814453125,
            0.43017578125,
            0.2071533203125,
            0.181396484375,
            1.25,
            1.00390625,
            1.666015625,
            1.341796875,
            2.09765625,
            0.83447265625,
            1.3134765625,
            0.2249755859375,
            0.031463623046875,
            0.0271759033203125,
            0.386962890625,
            0.302734375,
            1.0849609375,
            0.58544921875,
            1.2216796875,
            0.4033203125,
            0.45849609375,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors options.inputLayout=nhwc options.filterLayout=ohwi',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.056060791015625, 0.71142578125, 0.65283203125, 0.38623046875,
            0.386962890625, 0.9462890625, 0.095703125, 0.92333984375,
            0.63623046875
          ],
          'descriptor': {shape: [1, 3, 3, 1], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.861328125, 0.626953125, 0.63671875, 0.83837890625,
            0.11883544921875, 0.9921875, 0.32861328125, 0.8740234375,
            0.720703125, 0.97998046875, 0.06170654296875, 0.322021484375, 0.75,
            0.39306640625, 0.1380615234375, 0.283935546875, 0.423583984375,
            0.1448974609375
          ],
          'descriptor': {shape: [2, 3, 3, 1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'inputLayout': 'nhwc', 'filterLayout': 'ohwi'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.04827880859375,
            0.054931640625,
            0.64794921875,
            0.70068359375,
            1.0439453125,
            0.70166015625,
            0.8623046875,
            0.269287109375,
            0.415771484375,
            0.210205078125,
            0.379638671875,
            0.420654296875,
            1.1787109375,
            0.95849609375,
            1.9912109375,
            1.8525390625,
            1.623046875,
            0.53759765625,
            1.25,
            0.394775390625,
            0.424560546875,
            0.3994140625,
            1.5087890625,
            1.578125,
            3.287109375,
            2.12109375,
            2.56640625,
            1.1416015625,
            1.814453125,
            0.43017578125,
            0.2071533203125,
            0.181396484375,
            1.25,
            1.00390625,
            1.666015625,
            1.341796875,
            2.09765625,
            0.83447265625,
            1.3134765625,
            0.2249755859375,
            0.031463623046875,
            0.0271759033203125,
            0.386962890625,
            0.302734375,
            1.0849609375,
            0.58544921875,
            1.2216796875,
            0.4033203125,
            0.45849609375,
            0.0921630859375
          ],
          'descriptor': {shape: [1, 5, 5, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'convTranspose2d float16 4D input and filter tensors options.bias',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            0.1109619140625, 0.8681640625, 0.734375, 0.4306640625,
            0.59814453125, 0.12322998046875, 0.1611328125, 0.08837890625,
            0.291015625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.6162109375, 0.26220703125, 0.794921875, 0.873046875, 0.8310546875,
            0.85498046875, 0.55517578125, 0.84033203125, 0.85302734375
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        },
        'convTranspose2dBias': {
          'data': [0.45166015625],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'bias': 'convTranspose2dBias'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            0.52001953125, 1.015625,      1.2197265625,  1.333984375,
            1.03515625,    0.81396484375, 1.783203125,   2.484375,
            2.3125,        1.177734375,   0.98876953125, 2.00390625,
            2.986328125,   2.5703125,     1.4150390625,  0.83154296875,
            1.3564453125,  1.85546875,    1.3828125,     0.8056640625,
            0.541015625,   0.63623046875, 0.82470703125, 0.771484375,
            0.69970703125
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'convTranspose2d float16 4D input and filter tensors, both negative input tensor and options.bias',
    'graph': {
      'inputs': {
        'convTranspose2dInput': {
          'data': [
            -0.10888671875, -0.298095703125, -0.390869140625, -0.5625,
            -0.732421875, -0.84228515625, -0.305908203125, -0.9765625,
            -0.01416015625
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'}
        },
        'convTranspose2dFilter': {
          'data': [
            0.6162109375, 0.26220703125, 0.794921875, 0.873046875, 0.8310546875,
            0.85498046875, 0.55517578125, 0.84033203125, 0.85302734375
          ],
          'descriptor': {shape: [1, 1, 3, 3], dataType: 'float16'},
          'constant': true
        },
        'convTranspose2dBias': {
          'data': [-0.845703125],
          'descriptor': {shape: [1], dataType: 'float16'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'convTranspose2d',
        'arguments': [
          {'input': 'convTranspose2dInput'},
          {'filter': 'convTranspose2dFilter'},
          {'options': {'bias': 'convTranspose2dBias'}}
        ],
        'outputs': 'convTranspose2dOutput'
      }],
      'expectedOutputs': {
        'convTranspose2dOutput': {
          'data': [
            -0.91259765625, -1.0576171875, -1.2509765625, -1.185546875,
            -1.15625,       -1.287109375,  -1.794921875,  -2.685546875,
            -2.228515625,   -1.849609375,  -1.5859375,    -2.890625,
            -3.73828125,    -3.53515625,   -1.91015625,   -1.4248046875,
            -2.83203125,    -3.494140625,  -3.025390625,  -1.576171875,
            -1.015625,      -1.64453125,   -1.935546875,  -1.6904296875,
            -0.85791015625
          ],
          'descriptor': {shape: [1, 1, 5, 5], dataType: 'float16'}
        }
      }
    }
  }
];

if (navigator.ml) {
  convTranspose2dTests.forEach((test) => {
    webnn_conformance_test(buildAndExecuteGraph, getPrecisionTolerance, test);
  });
} else {
  test(() => assert_implements(navigator.ml, 'missing navigator.ml'));
}
