// META: title=test WebNN API triangular operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-triangular
// Given a 2-D tensor (matrix), return a 2-D tensor containing either the upper
// or lower triangular part of the input tensor. If the input tensor has greater
// than 2 dimensions it is treated as a batch of matrices and the result has the
// same shape.
//
// dictionary MLTriangularOptions {
//   boolean upper = true;
//   [EnforceRange] long diagonal = 0;
// };
//
// MLOperand triangular(
//     MLOperand input, optional MLTriangularOptions options = {});


const getTriangularPrecisionTolerance = () => {
  return {metricType: 'ULP', value: 0};
};

const triangularTests = [
  {
    'name': 'triangular float32 2D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,
            -86.21631622314453,
            50.38116455078125,
            -98.47772216796875,
            -94.51172637939453,
            -21.42218589782715,
            0,
            -50.355342864990234,
            -37.015724182128906,
            97.25071716308594,
            73.36241149902344,
            -75.12358856201172,
            0,
            0,
            -29.916187286376953,
            67.48285675048828,
            42.132598876953125,
            -70.27587127685547,
            0,
            0,
            0,
            -88.97057342529297,
            77.58269500732422,
            91.03327178955078
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 3D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,
            -86.21631622314453,
            50.38116455078125,
            -98.47772216796875,
            0,
            -21.42218589782715,
            24.678754806518555,
            -50.355342864990234,
            0,
            0,
            73.36241149902344,
            -75.12358856201172,
            41.64348602294922,
            58.862911224365234,
            -29.916187286376953,
            67.48285675048828,
            0,
            -70.27587127685547,
            20.88446617126465,
            71.37139892578125,
            0,
            0,
            77.58269500732422,
            91.03327178955078
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,    -86.21631622314453,
            50.38116455078125,   0,
            -94.51172637939453,  -21.42218589782715,
            24.678754806518555,  -50.355342864990234,
            -37.015724182128906, 0,
            73.36241149902344,   -75.12358856201172,
            41.64348602294922,   58.862911224365234,
            -29.916187286376953, 0,
            42.132598876953125,  -70.27587127685547,
            20.88446617126465,   71.37139892578125,
            -84.96932220458984,  0,
            77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 5D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 1, 4, 1, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 1, 4, 1, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor explict options.upper=true',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'upper': true}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,    -86.21631622314453,
            50.38116455078125,   0,
            -94.51172637939453,  -21.42218589782715,
            24.678754806518555,  -50.355342864990234,
            -37.015724182128906, 0,
            73.36241149902344,   -75.12358856201172,
            41.64348602294922,   58.862911224365234,
            -29.916187286376953, 0,
            42.132598876953125,  -70.27587127685547,
            20.88446617126465,   71.37139892578125,
            -84.96932220458984,  0,
            77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor options.upper=false',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'upper': false}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,
            0,
            0,
            -98.47772216796875,
            -94.51172637939453,
            0,
            24.678754806518555,
            0,
            0,
            97.25071716308594,
            73.36241149902344,
            0,
            41.64348602294922,
            0,
            0,
            67.48285675048828,
            42.132598876953125,
            0,
            20.88446617126465,
            0,
            0,
            -88.97057342529297,
            77.58269500732422,
            0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor explict options.diagonal=0',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': 0}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,    -86.21631622314453,
            50.38116455078125,   0,
            -94.51172637939453,  -21.42218589782715,
            24.678754806518555,  -50.355342864990234,
            -37.015724182128906, 0,
            73.36241149902344,   -75.12358856201172,
            41.64348602294922,   58.862911224365234,
            -29.916187286376953, 0,
            42.132598876953125,  -70.27587127685547,
            20.88446617126465,   71.37139892578125,
            -84.96932220458984,  0,
            77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor options.diagonal=1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': 1}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0,
            -86.21631622314453,
            50.38116455078125,
            0,
            0,
            -21.42218589782715,
            0,
            -50.355342864990234,
            -37.015724182128906,
            0,
            0,
            -75.12358856201172,
            0,
            58.862911224365234,
            -29.916187286376953,
            0,
            0,
            -70.27587127685547,
            0,
            71.37139892578125,
            -84.96932220458984,
            0,
            0,
            91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor options.diagonal=-1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': -1}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor fully zero options.diagonal=3',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': 3}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'triangular float32 4D tensor fully copied options.diagonal=-2',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': -2}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'triangular float32 4D tensor options.upper=true options.diagonal=1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': true, 'diagonal': 1}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0,
            -86.21631622314453,
            50.38116455078125,
            0,
            0,
            -21.42218589782715,
            0,
            -50.355342864990234,
            -37.015724182128906,
            0,
            0,
            -75.12358856201172,
            0,
            58.862911224365234,
            -29.916187286376953,
            0,
            0,
            -70.27587127685547,
            0,
            71.37139892578125,
            -84.96932220458984,
            0,
            0,
            91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'triangular float32 4D tensor options.upper=false options.diagonal=1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': 1}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  0,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, 0,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  0,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   0,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'triangular float32 4D tensor options.upper=false options.diagonal=-1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': -1}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, 0, 0, -98.47772216796875, 0, 0,
            0, 0, 0, 97.25071716308594,  0, 0,
            0, 0, 0, 67.48285675048828,  0, 0,
            0, 0, 0, -88.97057342529297, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'triangular float32 4D tensor fully copied options.upper=false options.diagonal=3',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': 3}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'triangular float32 4D tensor fully zero options.upper=false options.diagonal=-2',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9194564819336,   -86.21631622314453,  50.38116455078125,
            -98.47772216796875, -94.51172637939453,  -21.42218589782715,
            24.678754806518555, -50.355342864990234, -37.015724182128906,
            97.25071716308594,  73.36241149902344,   -75.12358856201172,
            41.64348602294922,  58.862911224365234,  -29.916187286376953,
            67.48285675048828,  42.132598876953125,  -70.27587127685547,
            20.88446617126465,  71.37139892578125,   -84.96932220458984,
            -88.97057342529297, 77.58269500732422,   91.03327178955078
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': -2}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float32'}
        }
      }
    }
  },

  // float16 tests
  {
    'name': 'triangular float16 2D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375, -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            0,       -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            0,       0,         -29.921875, 67.5,  42.125,  -70.25,
            0,       0,         0,          -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 3D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, 0,       -21.421875,
            24.671875, -50.34375, 0,          0,     73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  0,       -70.25,
            20.890625, 71.375,    0,          0,     77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 3, 4], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     0, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  0, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 0, 42.125,  -70.25,
            20.890625, 71.375,    -85,        0, 77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 5D tensor default options',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 1, 4, 1, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [{'input': 'triangularInput'}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 1, 4, 1, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor explict options.upper=true',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'upper': true}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     0, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  0, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 0, 42.125,  -70.25,
            20.890625, 71.375,    -85,        0, 77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor options.upper=false',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'upper': false}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375, 0,     0,         -98.5, -94.5,    0,   24.671875, 0,
            0,       97.25, 73.375,    0,     41.65625, 0,   0,         67.5,
            42.125,  0,     20.890625, 0,     0,        -89, 77.5625,   0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor explict options.diagonal=0',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': 0}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     0, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  0, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 0, 42.125,  -70.25,
            20.890625, 71.375,    -85,        0, 77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor options.diagonal=1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': 1}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, -86.1875,  50.375,     0, 0, -21.421875,
            0, -50.34375, -37.03125,  0, 0, -75.125,
            0, 58.875,    -29.921875, 0, 0, -70.25,
            0, 71.375,    -85,        0, 0, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor options.diagonal=-1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': -1}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor fully zero options.diagonal=3',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': 3}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'triangular float16 4D tensor fully copied options.diagonal=-2',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments':
            [{'input': 'triangularInput'}, {'options': {'diagonal': -2}}],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'triangular float16 4D tensor options.upper=true options.diagonal=1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': true, 'diagonal': 1}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, -86.1875,  50.375,     0, 0, -21.421875,
            0, -50.34375, -37.03125,  0, 0, -75.125,
            0, 58.875,    -29.921875, 0, 0, -70.25,
            0, 71.375,    -85,        0, 0, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'triangular float16 4D tensor options.upper=false options.diagonal=1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': 1}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  0, -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, 0, 97.25, 73.375,  -75.125,
            41.65625,  58.875,    0, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    0, -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'triangular float16 4D tensor options.upper=false options.diagonal=-1',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': -1}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, 0, 0, -98.5, 0, 0, 0, 0, 0, 97.25, 0, 0,
            0, 0, 0, 67.5,  0, 0, 0, 0, 0, -89,   0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'triangular float16 4D tensor fully copied options.upper=false options.diagonal=3',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': 3}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'triangular float16 4D tensor fully zero options.upper=false options.diagonal=-2',
    'graph': {
      'inputs': {
        'triangularInput': {
          'data': [
            84.9375,   -86.1875,  50.375,     -98.5, -94.5,   -21.421875,
            24.671875, -50.34375, -37.03125,  97.25, 73.375,  -75.125,
            41.65625,  58.875,    -29.921875, 67.5,  42.125,  -70.25,
            20.890625, 71.375,    -85,        -89,   77.5625, 91.0625
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'triangular',
        'arguments': [
          {'input': 'triangularInput'},
          {'options': {'upper': false, 'diagonal': -2}}
        ],
        'outputs': 'triangularOutput'
      }],
      'expectedOutputs': {
        'triangularOutput': {
          'data': [
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0
          ],
          'descriptor': {shape: [2, 2, 2, 3], dataType: 'float16'}
        }
      }
    }
  }
];

if (navigator.ml) {
  triangularTests.forEach((test) => {
    webnn_conformance_test(
        buildAndExecuteGraph, getTriangularPrecisionTolerance, test);
  });
} else {
  test(() => assert_implements(navigator.ml, 'missing navigator.ml'));
}
