// META: title=test WebNN API concat operation
// META: global=window
// META: variant=?cpu
// META: variant=?gpu
// META: variant=?npu
// META: script=../resources/utils.js
// META: timeout=long

'use strict';

// https://www.w3.org/TR/webnn/#api-mlgraphbuilder-concat
// Concatenates the input tensors along a given axis.
//
// MLOperand concat(
//     sequence<MLOperand> inputs, [EnforceRange] unsigned long axis);


const getConcatPrecisionTolerance = (graphResources) => {
  const toleranceValueDict = {float32: 0, float16: 0};
  const expectedDataType =
      getExpectedDataTypeOfSingleOutput(graphResources.expectedOutputs);
  return {metricType: 'ULP', value: toleranceValueDict[expectedDataType]};
};

const concatTests = [
  {
    'name': 'concat two float32 1D constant tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [12], dataType: 'float32'},
          'constant': true
        },
        'concatInput2': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [12], dataType: 'float32'},
          'constant': true
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat two float32 1D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [12], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [12], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat two float16 1D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [12], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [12], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat three float32 1D tensors of different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407
          ],
          'descriptor': {shape: [4], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.576785683631897, 0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [8], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [12], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat three float16 1D tensors of different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407
          ],
          'descriptor': {shape: [4], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.576785683631897, 0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [8], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [12], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat four float32 1D tensors of same 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat four float16 1D tensors of same 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [6], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [6], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802
          ],
          'descriptor': {shape: [6], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat four float32 1D tensors of different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989],
          'descriptor': {shape: [2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.337996244430542, -0.990639865398407, 0.576785683631897,
            0.32276400923728943
          ],
          'descriptor': {shape: [4], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957
          ],
          'descriptor': {shape: [8], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            -0.3418811559677124, -0.9158143401145935, -0.7206121683120728,
            -0.7993468642234802, 0.6653800010681152, 0.03886038810014725,
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [10], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [24], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat four float16 1D tensors of different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989],
          'descriptor': {shape: [2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.337996244430542, -0.990639865398407, 0.576785683631897,
            0.32276400923728943
          ],
          'descriptor': {shape: [4], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957
          ],
          'descriptor': {shape: [8], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            -0.3418811559677124, -0.9158143401145935, -0.7206121683120728,
            -0.7993468642234802, 0.6653800010681152, 0.03886038810014725,
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [10], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [24], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat two float32 2D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [2, 6], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat two float16 2D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [2, 6], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 2D tensors of same others dimensions except different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 6], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 6], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 2D tensors of same others dimensions except different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 6], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 6], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat four float32 2D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [3, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [3, 2], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802
          ],
          'descriptor': {shape: [3, 2], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [12, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat four float16 2D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [3, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [3, 2], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802
          ],
          'descriptor': {shape: [3, 2], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [12, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 2D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725
          ],
          'descriptor': {shape: [2, 10], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [2, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 1}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,   0.337996244430542,
            -0.990639865398407,   0.576785683631897,   0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958,   0.5182055234909058,  -0.8742017149925232,
            -0.9531654119491577,  -0.6731740236282349, 0.49189892411231995,
            -0.15864109992980957, -0.3418811559677124, -0.9158143401145935,
            -0.7206121683120728,  -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725,  -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 12], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 2D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725
          ],
          'descriptor': {shape: [2, 10], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [2, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 1}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,   0.86181640625,     0.337890625,    -0.99072265625,
            0.57666015625, 0.32275390625,     -0.447265625,   0.11029052734375,
            -0.5947265625, -0.40283203125,    0.51806640625,  -0.8740234375,
            -0.953125,     -0.67333984375,    0.491943359375, -0.15869140625,
            -0.341796875,  -0.916015625,      -0.720703125,   -0.79931640625,
            0.66552734375, 0.038848876953125, -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [2, 12], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat three float32 2D tensors of same shape along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [4, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935
          ],
          'descriptor': {shape: [4, 2], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.5182055234909058, -0.8742017149925232,
            -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [4, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,   -0.5945112705230713,
            -0.402848482131958,   -0.7206121683120728, -0.7993468642234802,
            0.337996244430542,    -0.990639865398407,  -0.9531654119491577,
            -0.6731740236282349,  0.6653800010681152,  0.03886038810014725,
            0.576785683631897,    0.32276400923728943, 0.49189892411231995,
            -0.15864109992980957, 0.5182055234909058,  -0.8742017149925232,
            -0.44735023379325867, 0.11028251051902771, -0.3418811559677124,
            -0.9158143401145935,  -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [4, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat three float16 2D tensors of same shape along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [4, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935
          ],
          'descriptor': {shape: [4, 2], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.5182055234909058, -0.8742017149925232,
            -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [4, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,   0.86181640625,  -0.5947265625,  -0.40283203125,
            -0.720703125,  -0.79931640625, 0.337890625,    -0.99072265625,
            -0.953125,     -0.67333984375, 0.66552734375,  0.038848876953125,
            0.57666015625, 0.32275390625,  0.491943359375, -0.15869140625,
            0.51806640625, -0.8740234375,  -0.447265625,   0.11029052734375,
            -0.341796875,  -0.916015625,   -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [4, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat four float32 2D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989, 0.337996244430542],
          'descriptor': {shape: [3, 1], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713
          ],
          'descriptor': {shape: [3, 2], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124
          ],
          'descriptor': {shape: [3, 2], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563, -0.990639865398407,   0.576785683631897,
            -0.402848482131958,  -0.9531654119491577,  -0.9158143401145935,
            -0.7206121683120728, -0.7993468642234802,  0.861982524394989,
            0.32276400923728943, -0.44735023379325867, -0.6731740236282349,
            0.49189892411231995, 0.6653800010681152,   0.03886038810014725,
            0.5182055234909058,  0.337996244430542,    0.11028251051902771,
            -0.5945112705230713, -0.15864109992980957, -0.3418811559677124,
            -0.8742017149925232, -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [3, 8], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat four float16 2D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989, 0.337996244430542],
          'descriptor': {shape: [3, 1], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713
          ],
          'descriptor': {shape: [3, 2], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124
          ],
          'descriptor': {shape: [3, 2], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    -0.99072265625, 0.57666015625,
            -0.40283203125, -0.953125,      -0.916015625,
            -0.720703125,   -0.79931640625, 0.86181640625,
            0.32275390625,  -0.447265625,   -0.67333984375,
            0.491943359375, 0.66552734375,  0.038848876953125,
            0.51806640625,  0.337890625,    0.11029052734375,
            -0.5947265625,  -0.15869140625, -0.341796875,
            -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [3, 8], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 3D tensors of same others dimensions except different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [2, 1, 3], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [6, 1, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [8, 1, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 3D tensors of same others dimensions except different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [2, 1, 3], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [6, 1, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [8, 1, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat four float32 3D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989, 0.337996244430542],
          'descriptor': {shape: [3, 1, 1], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713
          ],
          'descriptor': {shape: [3, 2, 1], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124
          ],
          'descriptor': {shape: [3, 2, 1], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 3, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563, -0.990639865398407,   0.576785683631897,
            -0.402848482131958,  -0.9531654119491577,  -0.9158143401145935,
            -0.7206121683120728, -0.7993468642234802,  0.861982524394989,
            0.32276400923728943, -0.44735023379325867, -0.6731740236282349,
            0.49189892411231995, 0.6653800010681152,   0.03886038810014725,
            0.5182055234909058,  0.337996244430542,    0.11028251051902771,
            -0.5945112705230713, -0.15864109992980957, -0.3418811559677124,
            -0.8742017149925232, -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [3, 8, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat four float16 3D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989, 0.337996244430542],
          'descriptor': {shape: [3, 1, 1], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713
          ],
          'descriptor': {shape: [3, 2, 1], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124
          ],
          'descriptor': {shape: [3, 2, 1], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 3, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    -0.99072265625, 0.57666015625,
            -0.40283203125, -0.953125,      -0.916015625,
            -0.720703125,   -0.79931640625, 0.86181640625,
            0.32275390625,  -0.447265625,   -0.67333984375,
            0.491943359375, 0.66552734375,  0.038848876953125,
            0.51806640625,  0.337890625,    0.11029052734375,
            -0.5947265625,  -0.15869140625, -0.341796875,
            -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [3, 8, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat three float32 3D tensors of same shape along axis 2',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [2, 2, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935
          ],
          'descriptor': {shape: [2, 2, 2], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.5182055234909058, -0.8742017149925232,
            -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 2, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 2}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,   -0.5945112705230713,
            -0.402848482131958,   -0.7206121683120728, -0.7993468642234802,
            0.337996244430542,    -0.990639865398407,  -0.9531654119491577,
            -0.6731740236282349,  0.6653800010681152,  0.03886038810014725,
            0.576785683631897,    0.32276400923728943, 0.49189892411231995,
            -0.15864109992980957, 0.5182055234909058,  -0.8742017149925232,
            -0.44735023379325867, 0.11028251051902771, -0.3418811559677124,
            -0.9158143401145935,  -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 2, 6], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat three float16 3D tensors of same shape along axis 2',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [2, 2, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935
          ],
          'descriptor': {shape: [2, 2, 2], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.5182055234909058, -0.8742017149925232,
            -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 2, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 2}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,   0.86181640625,  -0.5947265625,  -0.40283203125,
            -0.720703125,  -0.79931640625, 0.337890625,    -0.99072265625,
            -0.953125,     -0.67333984375, 0.66552734375,  0.038848876953125,
            0.57666015625, 0.32275390625,  0.491943359375, -0.15869140625,
            0.51806640625, -0.8740234375,  -0.447265625,   0.11029052734375,
            -0.341796875,  -0.916015625,   -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [2, 2, 6], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 4D tensors of same others dimensions except different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 3, 1, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 3, 1, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [4, 3, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 4D tensors of same others dimensions except different 1st dimension along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 3, 1, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 3, 1, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 0}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [4, 3, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat three float32 4D tensors of same shape along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.5182055234909058, -0.8742017149925232,
            -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563, 0.861982524394989,   0.337996244430542,
            -0.990639865398407,  -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349, -0.7206121683120728,
            -0.7993468642234802, 0.6653800010681152,  0.03886038810014725,
            0.576785683631897,   0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 6, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat three float16 4D tensors of same shape along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.5182055234909058, -0.8742017149925232,
            -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [2, 2, 1, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 1}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [2, 6, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat three float32 4D tensors of same others dimensions except different 3rd dimension along axis 2',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407
          ],
          'descriptor': {shape: [1, 2, 2, 1], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.576785683631897, 0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349, 0.49189892411231995,
            -0.15864109992980957, -0.3418811559677124, -0.9158143401145935,
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725
          ],
          'descriptor': {shape: [1, 2, 8, 1], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 2, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 2}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.576785683631897,
            0.32276400923728943,  -0.44735023379325867, 0.11028251051902771,
            -0.5945112705230713,  -0.402848482131958,   -0.9531654119491577,
            -0.6731740236282349,  0.5182055234909058,   -0.8742017149925232,
            0.337996244430542,    -0.990639865398407,   0.49189892411231995,
            -0.15864109992980957, -0.3418811559677124,  -0.9158143401145935,
            -0.7206121683120728,  -0.7993468642234802,  0.6653800010681152,
            0.03886038810014725,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 12, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat three float16 4D tensors of same others dimensions except different 3rd dimension along axis 2',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407
          ],
          'descriptor': {shape: [1, 2, 2, 1], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.576785683631897, 0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349, 0.49189892411231995,
            -0.15864109992980957, -0.3418811559677124, -0.9158143401145935,
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725
          ],
          'descriptor': {shape: [1, 2, 8, 1], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 2, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 2}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,   0.86181640625,     0.57666015625,  0.32275390625,
            -0.447265625,  0.11029052734375,  -0.5947265625,  -0.40283203125,
            -0.953125,     -0.67333984375,    0.51806640625,  -0.8740234375,
            0.337890625,   -0.99072265625,    0.491943359375, -0.15869140625,
            -0.341796875,  -0.916015625,      -0.720703125,   -0.79931640625,
            0.66552734375, 0.038848876953125, -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [1, 2, 12, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat four float32 4D tensors of same others dimensions except different 4th dimension along axis 3',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989, 0.337996244430542],
          'descriptor': {shape: [1, 3, 1, 1], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [-0.990639865398407, 0.576785683631897, 0.32276400923728943],
          'descriptor': {shape: [1, 3, 1, 1], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [1, 3, 1, 2], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 3, 1, 4], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 3}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563, -0.990639865398407,  -0.44735023379325867,
            0.11028251051902771, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935, 0.861982524394989,
            0.576785683631897,   -0.5945112705230713, -0.402848482131958,
            -0.7206121683120728, -0.7993468642234802, 0.6653800010681152,
            0.03886038810014725, 0.337996244430542,   0.32276400923728943,
            -0.9531654119491577, -0.6731740236282349, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 3, 1, 8], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat four float16 4D tensors of same others dimensions except different 4th dimension along axis 3',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [-0.3944413363933563, 0.861982524394989, 0.337996244430542],
          'descriptor': {shape: [1, 3, 1, 1], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [-0.990639865398407, 0.576785683631897, 0.32276400923728943],
          'descriptor': {shape: [1, 3, 1, 1], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [1, 3, 1, 2], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 3, 1, 4], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 3}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    -0.99072265625, -0.447265625,   0.11029052734375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            0.86181640625,  0.57666015625,  -0.5947265625,  -0.40283203125,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.337890625,    0.32275390625,  -0.953125,      -0.67333984375,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [1, 3, 1, 8], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name': 'concat four float32 5D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float32'}
        },
        'concatInput4': {
          'data': [
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [4, 2, 1, 1, 3], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name': 'concat four float16 5D tensors of same shape along axis 0',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float16'}
        },
        'concatInput4': {
          'data': [
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 1, 1, 3], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {
            'inputs':
                ['concatInput1', 'concatInput2', 'concatInput3', 'concatInput4']
          },
          {'axis': 0}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [4, 2, 1, 1, 3], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 5D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 2, 3, 1, 1], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 6, 3, 1, 1], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 1}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,    0.337996244430542,
            -0.990639865398407,   0.576785683631897,    0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771,  -0.5945112705230713,
            -0.402848482131958,   -0.9531654119491577,  -0.6731740236282349,
            0.49189892411231995,  -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935,  -0.7206121683120728,  -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [1, 8, 3, 1, 1], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 5D tensors of same others dimensions except different 2nd dimension along axis 1',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [1, 2, 3, 1, 1], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 6, 3, 1, 1], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 1}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [1, 8, 3, 1, 1], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat three float32 5D tensors of same others dimensions except different 3rd dimension along axis 2',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407
          ],
          'descriptor': {shape: [1, 2, 1, 1, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            0.576785683631897, 0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [1, 2, 2, 1, 2], dataType: 'float32'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 3, 1, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 2}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563, 0.861982524394989,    0.576785683631897,
            0.32276400923728943, -0.44735023379325867, 0.11028251051902771,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728,  -0.7993468642234802,
            0.337996244430542,   -0.990639865398407,   -0.5945112705230713,
            -0.402848482131958,  -0.9531654119491577,  -0.6731740236282349,
            0.6653800010681152,  0.03886038810014725,  0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998,  0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 6, 1, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat three float16 5D tensors of same others dimensions except different 3rd dimension along axis 2',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407
          ],
          'descriptor': {shape: [1, 2, 1, 1, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            0.576785683631897, 0.32276400923728943, -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349
          ],
          'descriptor': {shape: [1, 2, 2, 1, 2], dataType: 'float16'}
        },
        'concatInput3': {
          'data': [
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 3, 1, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments': [
          {'inputs': ['concatInput1', 'concatInput2', 'concatInput3']},
          {'axis': 2}
        ],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,   0.86181640625,    0.57666015625,  0.32275390625,
            -0.447265625,  0.11029052734375, 0.491943359375, -0.15869140625,
            -0.341796875,  -0.916015625,     -0.720703125,   -0.79931640625,
            0.337890625,   -0.99072265625,   -0.5947265625,  -0.40283203125,
            -0.953125,     -0.67333984375,   0.66552734375,  0.038848876953125,
            0.51806640625, -0.8740234375,    -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [1, 2, 6, 1, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 5D tensors of same others dimensions except different 4th dimension along axis 3',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [3, 1, 1, 1, 2], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 1, 1, 3, 2], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 3}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563, 0.861982524394989,   -0.44735023379325867,
            0.11028251051902771, -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577, -0.6731740236282349, 0.337996244430542,
            -0.990639865398407,  0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935, -0.7206121683120728,
            -0.7993468642234802, 0.576785683631897,   0.32276400923728943,
            0.6653800010681152,  0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 1, 1, 4, 2], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 5D tensors of same others dimensions except different 4th dimension along axis 3',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943
          ],
          'descriptor': {shape: [3, 1, 1, 1, 2], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.44735023379325867, 0.11028251051902771, -0.5945112705230713,
            -0.402848482131958, -0.9531654119491577, -0.6731740236282349,
            0.49189892411231995, -0.15864109992980957, -0.3418811559677124,
            -0.9158143401145935, -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152, 0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232, -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [3, 1, 1, 3, 2], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 3}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,   0.86181640625,  -0.447265625,   0.11029052734375,
            -0.5947265625, -0.40283203125, -0.953125,      -0.67333984375,
            0.337890625,   -0.99072265625, 0.491943359375, -0.15869140625,
            -0.341796875,  -0.916015625,   -0.720703125,   -0.79931640625,
            0.57666015625, 0.32275390625,  0.66552734375,  0.038848876953125,
            0.51806640625, -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [3, 1, 1, 4, 2], dataType: 'float16'}
        }
      }
    }
  },
  {
    'name':
        'concat two float32 5D tensors of same others dimensions except different 5th dimension along axis 4',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [1, 2, 1, 1, 4], dataType: 'float32'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935, -0.7206121683120728,
            -0.7993468642234802, 0.6653800010681152, 0.03886038810014725,
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 1, 1, 8], dataType: 'float32'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 4}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.3944413363933563,  0.861982524394989,   0.337996244430542,
            -0.990639865398407,   -0.5945112705230713, -0.402848482131958,
            -0.9531654119491577,  -0.6731740236282349, 0.49189892411231995,
            -0.15864109992980957, -0.3418811559677124, -0.9158143401145935,
            0.576785683631897,    0.32276400923728943, -0.44735023379325867,
            0.11028251051902771,  -0.7206121683120728, -0.7993468642234802,
            0.6653800010681152,   0.03886038810014725, 0.5182055234909058,
            -0.8742017149925232,  -0.4790218770503998, 0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 1, 1, 12], dataType: 'float32'}
        }
      }
    }
  },
  {
    'name':
        'concat two float16 5D tensors of same others dimensions except different 5th dimension along axis 4',
    'graph': {
      'inputs': {
        'concatInput1': {
          'data': [
            -0.3944413363933563, 0.861982524394989, 0.337996244430542,
            -0.990639865398407, 0.576785683631897, 0.32276400923728943,
            -0.44735023379325867, 0.11028251051902771
          ],
          'descriptor': {shape: [1, 2, 1, 1, 4], dataType: 'float16'}
        },
        'concatInput2': {
          'data': [
            -0.5945112705230713, -0.402848482131958, -0.9531654119491577,
            -0.6731740236282349, 0.49189892411231995, -0.15864109992980957,
            -0.3418811559677124, -0.9158143401145935, -0.7206121683120728,
            -0.7993468642234802, 0.6653800010681152, 0.03886038810014725,
            0.5182055234909058, -0.8742017149925232, -0.4790218770503998,
            0.1211843192577362
          ],
          'descriptor': {shape: [1, 2, 1, 1, 8], dataType: 'float16'}
        }
      },
      'operators': [{
        'name': 'concat',
        'arguments':
            [{'inputs': ['concatInput1', 'concatInput2']}, {'axis': 4}],
        'outputs': 'concatOutput'
      }],
      'expectedOutputs': {
        'concatOutput': {
          'data': [
            -0.39453125,    0.86181640625,  0.337890625,    -0.99072265625,
            -0.5947265625,  -0.40283203125, -0.953125,      -0.67333984375,
            0.491943359375, -0.15869140625, -0.341796875,   -0.916015625,
            0.57666015625,  0.32275390625,  -0.447265625,   0.11029052734375,
            -0.720703125,   -0.79931640625, 0.66552734375,  0.038848876953125,
            0.51806640625,  -0.8740234375,  -0.47900390625, 0.12115478515625
          ],
          'descriptor': {shape: [1, 2, 1, 1, 12], dataType: 'float16'}
        }
      }
    }
  }
];

if (navigator.ml) {
  concatTests.forEach((test) => {
    webnn_conformance_test(
      buildAndExecuteGraph, getConcatPrecisionTolerance, test);
  });
} else {
  test(() => assert_implements(navigator.ml, 'missing navigator.ml'));
}
